<?php
// Global error handler - Fatal error'ları yakalamak için
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        header('Content-Type: application/json; charset=utf-8');
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'PHP Fatal Error: ' . $error['message'] . ' in ' . $error['file'] . ' on line ' . $error['line']
        ], JSON_UNESCAPED_UNICODE);
        exit();
    }
});

// Hata yakalama için try-catch
try {
    require_once '../config.php';
} catch (Exception $e) {
    header('Content-Type: application/json; charset=utf-8');
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Config yükleme hatası: ' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Sadece POST isteği kabul edilir');
}

// Veritabanı tablolarını kontrol et
$table_check = $conn->query("SHOW TABLES LIKE 'friendships'");
if ($table_check->num_rows === 0) {
    sendResponse(false, 'friendships tablosu bulunamadı. Lütfen database/schema.sql dosyasını çalıştırın.');
}

// user_mutes tablosu kontrolü (mute işlemleri için)
$table_check = $conn->query("SHOW TABLES LIKE 'user_mutes'");
if ($table_check->num_rows === 0) {
    // Tablo yoksa oluşturmayı dene (migration dosyasından)
    $create_table_sql = "
        CREATE TABLE IF NOT EXISTS `user_mutes` (
          `id` INT(11) NOT NULL AUTO_INCREMENT,
          `user_id` INT(11) NOT NULL COMMENT 'Susturan kullanıcı',
          `muted_user_id` INT(11) NOT NULL COMMENT 'Susturulan kullanıcı',
          `mute_until` DATETIME NULL COMMENT 'Susturma bitiş zamanı (NULL ise süresiz)',
          `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
          PRIMARY KEY (`id`),
          UNIQUE KEY `unique_user_mute` (`user_id`, `muted_user_id`),
          FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
          FOREIGN KEY (`muted_user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
          INDEX `idx_user_id` (`user_id`),
          INDEX `idx_muted_user_id` (`muted_user_id`),
          INDEX `idx_mute_until` (`mute_until`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";
    if (!$conn->query($create_table_sql)) {
        // Tablo oluşturulamadı, hata mesajı göster ama devam et (eski versiyonlar için)
        error_log('user_mutes tablosu oluşturulamadı: ' . $conn->error);
    }
}

try {
    $input = getJsonInput();
    if ($input === null) {
        sendResponse(false, 'JSON verisi alınamadı');
    }
    
    $action = sanitize($input['action'] ?? ''); // 'search', 'send_request', 'accept', 'reject', 'remove', 'list', 'list_pending'
    $user_id = intval($input['user_id'] ?? 0);

    if ($user_id <= 0) {
        sendResponse(false, 'Geçersiz kullanıcı ID');
    }

    // Kullanıcı kontrolü
    $user_check = $conn->prepare("SELECT id, username, email FROM users WHERE id = ?");
    if (!$user_check) {
        sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
    }
    $user_check->bind_param("i", $user_id);
    $user_check->execute();
    $user_result = $user_check->get_result();

    if ($user_result->num_rows === 0) {
        sendResponse(false, 'Kullanıcı bulunamadı');
    }

    if ($action === 'search') {
        // Kullanıcı ara
        $search_term = sanitize($input['search_term'] ?? '');
        if (empty($search_term)) {
            sendResponse(false, 'Arama terimi gereklidir');
        }
        
        $search_pattern = "%$search_term%";
        $stmt = $conn->prepare("
            SELECT u.id, u.username,
                   up.name as display_name, up.profile_image_data as profile_image
            FROM users u
            LEFT JOIN user_profiles up ON u.id = up.user_id
            WHERE (u.username LIKE ? OR u.email LIKE ?) 
            AND u.id != ?
            LIMIT 20
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("ssi", $search_pattern, $search_pattern, $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $users = [];
        while ($row = $result->fetch_assoc()) {
            // Arkadaşlık durumunu kontrol et
            $friendship_check = $conn->prepare("
                SELECT status FROM friendships 
                WHERE (user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?)
            ");
            if ($friendship_check) {
                $friend_id = $row['id'];
                $friendship_check->bind_param("iiii", $user_id, $friend_id, $friend_id, $user_id);
                $friendship_check->execute();
                $friendship_result = $friendship_check->get_result();
                
                $status = null;
                if ($friendship_result->num_rows > 0) {
                    $friendship = $friendship_result->fetch_assoc();
                    $status = $friendship['status'];
                }
            }
            
            $users[] = [
                'id' => $row['id'],
                'username' => $row['username'],
                'display_name' => $row['display_name'],
                'profile_image' => $row['profile_image'],
                'friendship_status' => $status ?? null
            ];
        }
        
        sendResponse(true, 'Kullanıcılar bulundu', ['users' => $users]);
        
    } elseif ($action === 'send_request') {
        // Arkadaşlık isteği gönder
        $friend_id = intval($input['friend_id'] ?? 0);
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        if ($friend_id === $user_id) {
            sendResponse(false, 'Kendinize arkadaşlık isteği gönderemezsiniz');
        }
        
        // Zaten arkadaş mı kontrol et
        $check = $conn->prepare("
            SELECT status FROM friendships 
            WHERE (user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?)
        ");
        if (!$check) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $check->bind_param("iiii", $user_id, $friend_id, $friend_id, $user_id);
        $check->execute();
        $check_result = $check->get_result();
        
        if ($check_result->num_rows > 0) {
            $existing = $check_result->fetch_assoc();
            if ($existing['status'] === 'accepted') {
                sendResponse(false, 'Zaten arkadaşsınız');
            } elseif ($existing['status'] === 'pending') {
                sendResponse(false, 'Zaten bekleyen bir istek var');
            }
        }
        
        // Arkadaşlık isteği gönder
        $stmt = $conn->prepare("INSERT INTO friendships (user_id, friend_id, status) VALUES (?, ?, 'pending')");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("ii", $user_id, $friend_id);
        
        if ($stmt->execute()) {
            sendResponse(true, 'Arkadaşlık isteği gönderildi');
        } else {
            sendResponse(false, 'Arkadaşlık isteği gönderilemedi: ' . $conn->error);
        }
        
    } elseif ($action === 'accept') {
        // Arkadaşlık isteğini kabul et
        $friend_id = intval($input['friend_id'] ?? 0);
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        // İsteği bul ve kabul et
        $stmt = $conn->prepare("
            UPDATE friendships 
            SET status = 'accepted' 
            WHERE user_id = ? AND friend_id = ? AND status = 'pending'
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("ii", $friend_id, $user_id);
        
        if ($stmt->execute() && $stmt->affected_rows > 0) {
            sendResponse(true, 'Arkadaşlık isteği kabul edildi');
        } else {
            sendResponse(false, 'Arkadaşlık isteği bulunamadı veya zaten işlenmiş');
        }
        
    } elseif ($action === 'reject') {
        // Arkadaşlık isteğini reddet
        $friend_id = intval($input['friend_id'] ?? 0);
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        // İsteği sil
        $stmt = $conn->prepare("
            DELETE FROM friendships 
            WHERE user_id = ? AND friend_id = ? AND status = 'pending'
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("ii", $friend_id, $user_id);
        
        if ($stmt->execute()) {
            sendResponse(true, 'Arkadaşlık isteği reddedildi');
        } else {
            sendResponse(false, 'İstek reddedilemedi: ' . $conn->error);
        }
        
    } elseif ($action === 'remove') {
        // Arkadaşlığı kaldır
        $friend_id = intval($input['friend_id'] ?? 0);
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        // Arkadaşlığı sil
        $stmt = $conn->prepare("
            DELETE FROM friendships 
            WHERE (user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?)
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("iiii", $user_id, $friend_id, $friend_id, $user_id);
        
        if ($stmt->execute()) {
            sendResponse(true, 'Arkadaşlık kaldırıldı');
        } else {
            sendResponse(false, 'Arkadaşlık kaldırılamadı: ' . $conn->error);
        }
        
    } elseif ($action === 'list') {
        // Arkadaş listesi
        $stmt = $conn->prepare("
            SELECT u.id, u.username, u.created_at, 
                   up.name as display_name, up.profile_image_data as profile_image
            FROM friendships f
            INNER JOIN users u ON (f.friend_id = u.id AND f.user_id = ?) OR (f.user_id = u.id AND f.friend_id = ?)
            LEFT JOIN user_profiles up ON u.id = up.user_id
            WHERE f.status = 'accepted'
            AND (f.user_id = ? OR f.friend_id = ?)
            ORDER BY f.updated_at DESC
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("iiii", $user_id, $user_id, $user_id, $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $friends = [];
        while ($row = $result->fetch_assoc()) {
            $friends[] = [
                'id' => $row['id'],
                'username' => $row['username'],
                'display_name' => $row['display_name'],
                'profile_image' => $row['profile_image'],
                'created_at' => $row['created_at']
            ];
        }
        
        sendResponse(true, 'Arkadaşlar listelendi', ['friends' => $friends]);
        
    } elseif ($action === 'list_pending') {
        // Bekleyen istekler
        $stmt = $conn->prepare("
            SELECT u.id, u.username, u.created_at,
                   up.name as display_name, up.profile_image_data as profile_image
            FROM friendships f
            INNER JOIN users u ON f.user_id = u.id
            LEFT JOIN user_profiles up ON u.id = up.user_id
            WHERE f.friend_id = ? AND f.status = 'pending'
            ORDER BY f.created_at DESC
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $requests = [];
        while ($row = $result->fetch_assoc()) {
            $requests[] = [
                'id' => $row['id'],
                'username' => $row['username'],
                'display_name' => $row['display_name'],
                'profile_image' => $row['profile_image'],
                'created_at' => $row['created_at']
            ];
        }
        
        sendResponse(true, 'Bekleyen istekler listelendi', ['requests' => $requests]);
        
    } elseif ($action === 'block') {
        // Kullanıcıyı engelle
        $friend_id = intval($input['friend_id'] ?? 0);
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        if ($friend_id === $user_id) {
            sendResponse(false, 'Kendinizi engelleyemezsiniz');
        }
        
        // Mevcut arkadaşlık durumunu kontrol et
        $check = $conn->prepare("
            SELECT id, status FROM friendships 
            WHERE (user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?)
        ");
        if (!$check) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $check->bind_param("iiii", $user_id, $friend_id, $friend_id, $user_id);
        $check->execute();
        $check_result = $check->get_result();
        
        if ($check_result->num_rows > 0) {
            // Mevcut kaydı güncelle
            $stmt = $conn->prepare("
                UPDATE friendships 
                SET status = 'blocked' 
                WHERE (user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?)
            ");
            if (!$stmt) {
                sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
            }
            $stmt->bind_param("iiii", $user_id, $friend_id, $friend_id, $user_id);
        } else {
            // Yeni kayıt oluştur
            $stmt = $conn->prepare("INSERT INTO friendships (user_id, friend_id, status) VALUES (?, ?, 'blocked')");
            if (!$stmt) {
                sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
            }
            $stmt->bind_param("ii", $user_id, $friend_id);
        }
        
        if ($stmt->execute()) {
            // Engelleme yapıldığında arkadaşlığı da kaldır (eğer varsa)
            $remove_friendship = $conn->prepare("
                DELETE FROM friendships 
                WHERE status = 'accepted' 
                AND ((user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?))
            ");
            if ($remove_friendship) {
                $remove_friendship->bind_param("iiii", $user_id, $friend_id, $friend_id, $user_id);
                $remove_friendship->execute();
            }
            
            sendResponse(true, 'Engellenenler');
        } else {
            sendResponse(false, 'Kullanıcı engellenemedi: ' . $conn->error);
        }
        
    } elseif ($action === 'unblock') {
        // Kullanıcının engelini kaldır
        $friend_id = intval($input['friend_id'] ?? 0);
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        // Engelleme kaydını sil
        $stmt = $conn->prepare("
            DELETE FROM friendships 
            WHERE ((user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?))
            AND status = 'blocked'
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("iiii", $user_id, $friend_id, $friend_id, $user_id);
        
        if ($stmt->execute() && $stmt->affected_rows > 0) {
            sendResponse(true, 'Kullanıcının engeli kaldırıldı');
        } else {
            sendResponse(false, 'Engel kaldırılamadı veya kayıt bulunamadı');
        }
        
    } elseif ($action === 'mute') {
        // Kullanıcıyı sustur
        $friend_id = intval($input['friend_id'] ?? 0);
        $duration = sanitize($input['duration'] ?? 'indefinite'); // '5dk', '10dk', '2 saat', '8 saat', 'indefinite'
        
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        if ($friend_id === $user_id) {
            sendResponse(false, 'Kendinizi susturamazsınız');
        }
        
        // Süre hesaplama
        $mute_until = null;
        if ($duration !== 'indefinite') {
            $minutes = 0;
            if ($duration === '5dk') {
                $minutes = 5;
            } elseif ($duration === '10dk') {
                $minutes = 10;
            } elseif ($duration === '2 saat') {
                $minutes = 120;
            } elseif ($duration === '8 saat') {
                $minutes = 480;
            }
            
            if ($minutes > 0) {
                $mute_until = date('Y-m-d H:i:s', strtotime("+{$minutes} minutes"));
            }
        }
        
        // Mevcut kaydı kontrol et
        $check = $conn->prepare("
            SELECT id FROM user_mutes 
            WHERE user_id = ? AND muted_user_id = ?
        ");
        if (!$check) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $check->bind_param("ii", $user_id, $friend_id);
        $check->execute();
        $check_result = $check->get_result();
        
        if ($check_result->num_rows > 0) {
            // Mevcut kaydı güncelle
            if ($mute_until) {
                $stmt = $conn->prepare("
                    UPDATE user_mutes 
                    SET mute_until = ? 
                    WHERE user_id = ? AND muted_user_id = ?
                ");
                if (!$stmt) {
                    sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
                }
                $stmt->bind_param("sii", $mute_until, $user_id, $friend_id);
            } else {
                $stmt = $conn->prepare("
                    UPDATE user_mutes 
                    SET mute_until = NULL 
                    WHERE user_id = ? AND muted_user_id = ?
                ");
                if (!$stmt) {
                    sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
                }
                $stmt->bind_param("ii", $user_id, $friend_id);
            }
        } else {
            // Yeni kayıt oluştur
            $stmt = $conn->prepare("
                INSERT INTO user_mutes (user_id, muted_user_id, mute_until) 
                VALUES (?, ?, ?)
            ");
            if (!$stmt) {
                sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
            }
            $stmt->bind_param("iis", $user_id, $friend_id, $mute_until);
        }
        
        if ($stmt->execute()) {
            $duration_text = $duration === 'indefinite' ? 'süresiz' : $duration;
            sendResponse(true, "Kullanıcı {$duration_text} boyunca susturuldu");
        } else {
            sendResponse(false, 'Kullanıcı susturulamadı: ' . $conn->error);
        }
        
    } elseif ($action === 'unmute') {
        // Kullanıcının susturmasını kaldır
        $friend_id = intval($input['friend_id'] ?? 0);
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        $stmt = $conn->prepare("
            DELETE FROM user_mutes 
            WHERE user_id = ? AND muted_user_id = ?
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("ii", $user_id, $friend_id);
        
        if ($stmt->execute() && $stmt->affected_rows > 0) {
            sendResponse(true, 'Kullanıcının susturması kaldırıldı');
        } else {
            sendResponse(false, 'Susturma kaldırılamadı veya kayıt bulunamadı');
        }
        
    } elseif ($action === 'list_blocked') {
        // Engellenen kullanıcıları listele
        $stmt = $conn->prepare("
            SELECT u.id, u.username, u.created_at,
                   up.name as display_name, up.profile_image_data as profile_image,
                   f.created_at as blocked_at
            FROM friendships f
            INNER JOIN users u ON (
                (f.user_id = ? AND f.friend_id = u.id) OR 
                (f.user_id = u.id AND f.friend_id = ?)
            )
            LEFT JOIN user_profiles up ON u.id = up.user_id
            WHERE f.status = 'blocked'
            AND f.user_id = ?
            ORDER BY f.updated_at DESC
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("iii", $user_id, $user_id, $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $blocked = [];
        while ($row = $result->fetch_assoc()) {
            $blocked[] = [
                'id' => $row['id'],
                'username' => $row['username'],
                'display_name' => $row['display_name'],
                'profile_image' => $row['profile_image'],
                'blocked_at' => $row['blocked_at']
            ];
        }
        
        sendResponse(true, 'Engellenen kullanıcılar listelendi', ['blocked' => $blocked]);
        
    } elseif ($action === 'check_mute_status') {
        // Kullanıcının susturulup susturulmadığını kontrol et
        $friend_id = intval($input['friend_id'] ?? 0);
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        $stmt = $conn->prepare("
            SELECT mute_until, created_at
            FROM user_mutes 
            WHERE user_id = ? AND muted_user_id = ?
            AND (mute_until IS NULL OR mute_until > NOW())
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("ii", $user_id, $friend_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $mute_data = $result->fetch_assoc();
            sendResponse(true, 'Kullanıcı susturulmuş', [
                'is_muted' => true,
                'mute_until' => $mute_data['mute_until'],
                'muted_at' => $mute_data['created_at']
            ]);
        } else {
            sendResponse(true, 'Kullanıcı susturulmamış', ['is_muted' => false]);
        }
        
    } else {
        sendResponse(false, 'Geçersiz işlem');
    }

} catch (Exception $e) {
    sendResponse(false, 'Hata: ' . $e->getMessage() . ' - Satır: ' . $e->getLine());
}

$conn->close();
?>

