<?php
// Global error handler - Fatal error'ları yakalamak için
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        header('Content-Type: application/json; charset=utf-8');
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'PHP Fatal Error: ' . $error['message'] . ' in ' . $error['file'] . ' on line ' . $error['line']
        ], JSON_UNESCAPED_UNICODE);
        exit();
    }
});

// Hata yakalama için try-catch
try {
    require_once '../config.php';
} catch (Exception $e) {
    header('Content-Type: application/json; charset=utf-8');
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Config yükleme hatası: ' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Sadece POST isteği kabul edilir');
}

// Veritabanı tablolarını kontrol et
try {
    $table_check = $conn->query("SHOW TABLES LIKE 'messages'");
    if (!$table_check) {
        sendResponse(false, 'Veritabanı hatası (messages tablosu kontrolü): ' . $conn->error);
    }
    if ($table_check->num_rows === 0) {
        sendResponse(false, 'messages tablosu bulunamadı. Lütfen database/schema.sql dosyasını çalıştırın.');
    }

    $table_check = $conn->query("SHOW TABLES LIKE 'friendships'");
    if (!$table_check) {
        sendResponse(false, 'Veritabanı hatası (friendships tablosu kontrolü): ' . $conn->error);
    }
    if ($table_check->num_rows === 0) {
        sendResponse(false, 'friendships tablosu bulunamadı. Lütfen database/schema.sql dosyasını çalıştırın.');
    }
} catch (Exception $e) {
    sendResponse(false, 'Tablo kontrolü hatası: ' . $e->getMessage());
}

try {
    $input = getJsonInput();
    if ($input === null) {
        sendResponse(false, 'JSON verisi alınamadı');
    }
    
    $action = sanitize($input['action'] ?? ''); // 'send', 'get_conversations', 'get_messages', 'mark_read'
    $user_id = intval($input['user_id'] ?? 0);

    if ($user_id <= 0) {
        sendResponse(false, 'Geçersiz kullanıcı ID');
    }

    // Kullanıcı kontrolü
    $user_check = $conn->prepare("SELECT id FROM users WHERE id = ?");
    if (!$user_check) {
        sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
    }
    $user_check->bind_param("i", $user_id);
    $user_check->execute();
    $user_result = $user_check->get_result();

    if ($user_result->num_rows === 0) {
        sendResponse(false, 'Kullanıcı bulunamadı');
    }

    if ($action === 'send') {
        // Mesaj gönder
        $receiver_id = intval($input['receiver_id'] ?? 0);
        $message = trim($input['message'] ?? '');
        
        if ($receiver_id <= 0) {
            sendResponse(false, 'Geçersiz alıcı ID');
        }
        
        if (empty($message)) {
            sendResponse(false, 'Mesaj boş olamaz');
        }
        
        if ($receiver_id === $user_id) {
            sendResponse(false, 'Kendinize mesaj gönderemezsiniz');
        }
        
        // Engellenmiş mi kontrol et
        $block_check = $conn->prepare("
            SELECT status FROM friendships 
            WHERE status = 'blocked' 
            AND ((user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?))
        ");
        if (!$block_check) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $block_check->bind_param("iiii", $user_id, $receiver_id, $receiver_id, $user_id);
        if (!$block_check->execute()) {
            sendResponse(false, 'Engelleme kontrolü hatası: ' . $block_check->error);
        }
        $block_result = $block_check->get_result();
        if (!$block_result) {
            sendResponse(false, 'Engelleme sonucu alınamadı: ' . $conn->error);
        }
        
        if ($block_result->num_rows > 0) {
            $block_row = $block_result->fetch_assoc();
            // Gönderen engellenmiş mi kontrol et
            $check_sender_blocked = $conn->prepare("
                SELECT status FROM friendships 
                WHERE status = 'blocked' 
                AND user_id = ? AND friend_id = ?
            ");
            if ($check_sender_blocked) {
                $check_sender_blocked->bind_param("ii", $receiver_id, $user_id);
                $check_sender_blocked->execute();
                $sender_blocked_result = $check_sender_blocked->get_result();
                if ($sender_blocked_result && $sender_blocked_result->num_rows > 0) {
                    sendResponse(false, 'Bu kullanıcı sizi engellemiş');
                }
            }
            sendResponse(false, 'Bu kullanıcıyı engellediniz');
        }
        
        // Arkadaş mı kontrol et
        $friendship_check = $conn->prepare("
            SELECT status FROM friendships 
            WHERE status = 'accepted' 
            AND ((user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?))
        ");
        if (!$friendship_check) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $friendship_check->bind_param("iiii", $user_id, $receiver_id, $receiver_id, $user_id);
        if (!$friendship_check->execute()) {
            sendResponse(false, 'Arkadaşlık kontrolü hatası: ' . $friendship_check->error);
        }
        $friendship_result = $friendship_check->get_result();
        if (!$friendship_result) {
            sendResponse(false, 'Arkadaşlık sonucu alınamadı: ' . $conn->error);
        }
        
        if ($friendship_result->num_rows === 0) {
            sendResponse(false, 'Sadece arkadaşlarınıza mesaj gönderebilirsiniz');
        }
        
        // Yanıtlanan mesaj ID'si
        $reply_to_message_id = isset($input['reply_to_message_id']) ? intval($input['reply_to_message_id']) : null;
        
        // Mesajı kaydet
        $stmt = $conn->prepare("INSERT INTO messages (sender_id, receiver_id, message, reply_to_message_id) VALUES (?, ?, ?, ?)");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("iisi", $user_id, $receiver_id, $message, $reply_to_message_id);
        
        if ($stmt->execute()) {
            $message_id = $conn->insert_id;
            
            // Push notification gönder (alıcının cihazlarına)
            try {
                // Alıcının push token'larını al
                $push_tokens_stmt = $conn->prepare("
                    SELECT push_token FROM device_sync 
                    WHERE user_id = ? AND push_token IS NOT NULL AND push_token != ''
                ");
                if ($push_tokens_stmt) {
                    $push_tokens_stmt->bind_param("i", $receiver_id);
                    $push_tokens_stmt->execute();
                    $push_tokens_result = $push_tokens_stmt->get_result();
                    
                    $pushTokens = [];
                    while ($row = $push_tokens_result->fetch_assoc()) {
                        if (!empty($row['push_token'])) {
                            $pushTokens[] = $row['push_token'];
                        }
                    }
                    
                    // Gönderenin bilgilerini al
                    $sender_info_stmt = $conn->prepare("
                        SELECT u.username, up.name as display_name
                        FROM users u
                        LEFT JOIN user_profiles up ON u.id = up.user_id
                        WHERE u.id = ?
                    ");
                    if ($sender_info_stmt && !empty($pushTokens)) {
                        $sender_info_stmt->bind_param("i", $user_id);
                        $sender_info_stmt->execute();
                        $sender_info_result = $sender_info_stmt->get_result();
                        if ($sender_info_result->num_rows > 0) {
                            $sender_info = $sender_info_result->fetch_assoc();
                            $senderName = $sender_info['display_name'] ?: $sender_info['username'];
                            
                            // Susturulmuş mu kontrol et
                            $mute_check = $conn->prepare("
                                SELECT id FROM user_mutes 
                                WHERE user_id = ? AND muted_user_id = ?
                                AND (mute_until IS NULL OR mute_until > NOW())
                            ");
                            $shouldNotify = true;
                            if ($mute_check) {
                                $mute_check->bind_param("ii", $receiver_id, $user_id);
                                $mute_check->execute();
                                $mute_result = $mute_check->get_result();
                                if ($mute_result->num_rows > 0) {
                                    $shouldNotify = false; // Susturulmuş, bildirim gönderme
                                }
                            }
                            
                            if ($shouldNotify) {
                                // Expo Push Notification gönder
                                $notificationData = [
                                    'type' => 'message',
                                    'senderId' => $user_id,
                                    'messageId' => $message_id,
                                ];
                                
                                $notificationBody = strlen($message) > 50 ? substr($message, 0, 50) . '...' : $message;
                                
                                // cURL ile Expo Push Notification Service'e istek gönder
                                $messages = [];
                                foreach ($pushTokens as $token) {
                                    $messages[] = [
                                        'to' => $token,
                                        'sound' => 'default',
                                        'title' => $senderName,
                                        'body' => $notificationBody,
                                        'data' => $notificationData,
                                        'badge' => 1,
                                    ];
                                }
                                
                                $ch = curl_init();
                                curl_setopt($ch, CURLOPT_URL, 'https://exp.host/--/api/v2/push/send');
                                curl_setopt($ch, CURLOPT_POST, true);
                                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                                    'Accept: application/json',
                                    'Content-Type: application/json',
                                ]);
                                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($messages));
                                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
                                curl_setopt($ch, CURLOPT_TIMEOUT, 5);
                                
                                // Async olarak gönder (blocking olmasın)
                                curl_exec($ch);
                                curl_close($ch);
                            }
                        }
                    }
                }
            } catch (Exception $pushError) {
                // Push notification hatası mesaj göndermeyi engellemesin
                error_log('Push notification error: ' . $pushError->getMessage());
            }
            
            // Gönderilen mesajı profil bilgileriyle birlikte döndür
            $msg_stmt = $conn->prepare("
                SELECT m.id, m.sender_id, m.receiver_id, m.message, m.is_read, m.created_at,
                       m.updated_at, m.is_edited, m.reply_to_message_id,
                       u.username as sender_username,
                       up.name as sender_display_name,
                       up.profile_image_data as sender_profile_image
                FROM messages m
                INNER JOIN users u ON m.sender_id = u.id
                LEFT JOIN user_profiles up ON u.id = up.user_id
                WHERE m.id = ?
            ");
            if ($msg_stmt) {
                $msg_stmt->bind_param("i", $message_id);
                $msg_stmt->execute();
                $msg_result = $msg_stmt->get_result();
                if ($msg_result && $msg_result->num_rows > 0) {
                    $msg_row = $msg_result->fetch_assoc();
                    sendResponse(true, 'Mesaj gönderildi', [
                        'message_id' => $message_id,
                        'message' => [
                            'id' => $msg_row['id'],
                            'sender_id' => $msg_row['sender_id'],
                            'receiver_id' => $msg_row['receiver_id'],
                            'message' => $msg_row['message'],
                            'is_read' => false,
                            'created_at' => $msg_row['created_at'],
                            'updated_at' => $msg_row['updated_at'],
                            'is_edited' => false,
                            'reply_to_message_id' => $msg_row['reply_to_message_id'],
                            'sender_username' => $msg_row['sender_username'],
                            'sender_display_name' => $msg_row['sender_display_name'],
                            'sender_profile_image' => $msg_row['sender_profile_image'],
                            'is_mine' => true
                        ]
                    ]);
                } else {
                    sendResponse(true, 'Mesaj gönderildi', ['message_id' => $message_id]);
                }
            } else {
                sendResponse(true, 'Mesaj gönderildi', ['message_id' => $message_id]);
            }
        } else {
            sendResponse(false, 'Mesaj gönderilemedi: ' . $conn->error);
        }
        
    } elseif ($action === 'get_conversations') {
        // Konuşma listesi (son mesajlar) - Susturma bilgisi ile
        $stmt = $conn->prepare("
            SELECT DISTINCT
                u.id as friend_id,
                u.username,
                up.name as display_name,
                up.profile_image_data as profile_image,
                (SELECT message FROM messages 
                 WHERE ((sender_id = ? AND receiver_id = u.id) OR (sender_id = u.id AND receiver_id = ?))
                 ORDER BY created_at DESC LIMIT 1) as last_message,
                (SELECT created_at FROM messages 
                 WHERE ((sender_id = ? AND receiver_id = u.id) OR (sender_id = u.id AND receiver_id = ?))
                 ORDER BY created_at DESC LIMIT 1) as last_message_time,
                (SELECT COUNT(*) FROM messages 
                 WHERE receiver_id = ? AND sender_id = u.id AND is_read = 0) as unread_count,
                (SELECT CASE 
                    WHEN mute_until IS NULL THEN 1
                    WHEN mute_until > NOW() THEN 1
                    ELSE 0
                 END FROM user_mutes 
                 WHERE user_id = ? AND muted_user_id = u.id) as is_muted
            FROM users u
            INNER JOIN friendships f ON (
                (f.user_id = ? AND f.friend_id = u.id) OR 
                (f.user_id = u.id AND f.friend_id = ?)
            )
            LEFT JOIN user_profiles up ON u.id = up.user_id
            WHERE f.status = 'accepted'
            AND (f.user_id = ? OR f.friend_id = ?)
            ORDER BY last_message_time DESC
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        // Parametreler: last_message (sender_id, receiver_id), last_message_time (sender_id, receiver_id), 
        // unread_count (receiver_id), is_muted (user_id), friendship (user_id, friend_id), WHERE (user_id, friend_id)
        // Toplam: 2 + 2 + 1 + 1 + 2 + 2 = 10 parametre
        $stmt->bind_param("iiiiiiiiii", $user_id, $user_id, $user_id, $user_id, $user_id, $user_id, $user_id, $user_id, $user_id, $user_id);
        if (!$stmt->execute()) {
            sendResponse(false, 'Sorgu hatası: ' . $stmt->error);
        }
        $result = $stmt->get_result();
        if (!$result) {
            sendResponse(false, 'Sonuç alınamadı: ' . $conn->error);
        }
        
        $conversations = [];
        while ($row = $result->fetch_assoc()) {
            $conversations[] = [
                'friend_id' => $row['friend_id'],
                'username' => $row['username'],
                'display_name' => $row['display_name'],
                'profile_image' => $row['profile_image'],
                'last_message' => $row['last_message'],
                'last_message_time' => $row['last_message_time'],
                'unread_count' => intval($row['unread_count'] ?? 0),
                'is_muted' => (bool)($row['is_muted'] ?? 0)
            ];
        }
        
        sendResponse(true, 'Konuşmalar listelendi', ['conversations' => $conversations]);
        
    } elseif ($action === 'get_messages') {
        // Belirli bir kullanıcıyla mesajlar
        $friend_id = intval($input['friend_id'] ?? 0);
        $limit = intval($input['limit'] ?? 50);
        $offset = intval($input['offset'] ?? 0);
        
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        // Arkadaş mı kontrol et
        $friendship_check = $conn->prepare("
            SELECT status FROM friendships 
            WHERE status = 'accepted' 
            AND ((user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?))
        ");
        if (!$friendship_check) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $friendship_check->bind_param("iiii", $user_id, $friend_id, $friend_id, $user_id);
        if (!$friendship_check->execute()) {
            sendResponse(false, 'Arkadaşlık kontrolü hatası: ' . $friendship_check->error);
        }
        $friendship_result = $friendship_check->get_result();
        if (!$friendship_result) {
            sendResponse(false, 'Arkadaşlık sonucu alınamadı: ' . $conn->error);
        }
        
        if ($friendship_result->num_rows === 0) {
            sendResponse(false, 'Sadece arkadaşlarınızla mesajlaşabilirsiniz');
        }
        
        // Mesajları getir (profil resmi ve yanıtlanan mesaj ile birlikte)
        $stmt = $conn->prepare("
            SELECT m.id, m.sender_id, m.receiver_id, m.message, m.is_read, m.created_at,
                   m.updated_at, m.is_edited, m.reply_to_message_id,
                   u.username as sender_username,
                   up.name as sender_display_name,
                   up.profile_image_data as sender_profile_image,
                   rm.message as reply_to_message_text,
                   ru.username as reply_to_sender_username,
                   rup.name as reply_to_sender_display_name
            FROM messages m
            INNER JOIN users u ON m.sender_id = u.id
            LEFT JOIN user_profiles up ON u.id = up.user_id
            LEFT JOIN messages rm ON m.reply_to_message_id = rm.id
            LEFT JOIN users ru ON rm.sender_id = ru.id
            LEFT JOIN user_profiles rup ON ru.id = rup.user_id
            WHERE (m.sender_id = ? AND m.receiver_id = ?) OR (m.sender_id = ? AND m.receiver_id = ?)
            ORDER BY m.created_at DESC
            LIMIT ? OFFSET ?
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("iiiiii", $user_id, $friend_id, $friend_id, $user_id, $limit, $offset);
        if (!$stmt->execute()) {
            sendResponse(false, 'Mesaj sorgusu hatası: ' . $stmt->error);
        }
        $result = $stmt->get_result();
        if (!$result) {
            sendResponse(false, 'Mesaj sonucu alınamadı: ' . $conn->error);
        }
        
        $messages = [];
        while ($row = $result->fetch_assoc()) {
            $message_data = [
                'id' => $row['id'],
                'sender_id' => $row['sender_id'],
                'receiver_id' => $row['receiver_id'],
                'message' => $row['message'],
                'is_read' => (bool)$row['is_read'],
                'created_at' => $row['created_at'],
                'updated_at' => $row['updated_at'],
                'is_edited' => (bool)$row['is_edited'],
                'reply_to_message_id' => $row['reply_to_message_id'],
                'sender_username' => $row['sender_username'],
                'sender_display_name' => $row['sender_display_name'],
                'sender_profile_image' => $row['sender_profile_image'],
                'is_mine' => ($row['sender_id'] === $user_id)
            ];
            
            // Yanıtlanan mesaj varsa ekle
            if ($row['reply_to_message_id']) {
                $message_data['reply_to_message'] = [
                    'message' => $row['reply_to_message_text'],
                    'sender_username' => $row['reply_to_sender_username'],
                    'sender_display_name' => $row['reply_to_sender_display_name']
                ];
            } else {
                $message_data['reply_to_message'] = null;
            }
            
            $messages[] = $message_data;
        }
        
        // Mesajları okundu olarak işaretle
        $update_stmt = $conn->prepare("
            UPDATE messages 
            SET is_read = 1 
            WHERE receiver_id = ? AND sender_id = ? AND is_read = 0
        ");
        if ($update_stmt) {
            $update_stmt->bind_param("ii", $user_id, $friend_id);
            $update_stmt->execute();
        }
        
        // Mesajları eskiden yeniye sırala (en yeni en altta olsun)
        sendResponse(true, 'Mesajlar getirildi', ['messages' => array_reverse($messages)]);
        
    } elseif ($action === 'mark_read') {
        // Mesajları okundu olarak işaretle
        $friend_id = intval($input['friend_id'] ?? 0);
        
        if ($friend_id <= 0) {
            sendResponse(false, 'Geçersiz arkadaş ID');
        }
        
        $stmt = $conn->prepare("
            UPDATE messages 
            SET is_read = 1 
            WHERE receiver_id = ? AND sender_id = ? AND is_read = 0
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("ii", $user_id, $friend_id);
        
        if ($stmt->execute()) {
            sendResponse(true, 'Mesajlar okundu olarak işaretlendi');
        } else {
            sendResponse(false, 'Mesajlar güncellenemedi: ' . $conn->error);
        }
        
    } elseif ($action === 'delete') {
        // Mesaj sil
        $message_id = intval($input['message_id'] ?? 0);
        
        if ($message_id <= 0) {
            sendResponse(false, 'Geçersiz mesaj ID');
        }
        
        // Sadece gönderen silebilir
        $stmt = $conn->prepare("
            DELETE FROM messages 
            WHERE id = ? AND sender_id = ?
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("ii", $message_id, $user_id);
        
        if ($stmt->execute() && $stmt->affected_rows > 0) {
            sendResponse(true, 'Mesaj silindi');
        } else {
            sendResponse(false, 'Mesaj silinemedi veya bulunamadı');
        }
        
    } elseif ($action === 'edit') {
        // Mesaj düzenle (24 saat içinde)
        $message_id = intval($input['message_id'] ?? 0);
        $new_message = sanitize($input['message'] ?? '');
        
        if ($message_id <= 0) {
            sendResponse(false, 'Geçersiz mesaj ID');
        }
        
        if (empty($new_message)) {
            sendResponse(false, 'Mesaj boş olamaz');
        }
        
        // Mesajı kontrol et - sadece gönderen düzenleyebilir ve 24 saat içinde olmalı
        $check_stmt = $conn->prepare("
            SELECT created_at, sender_id 
            FROM messages 
            WHERE id = ? AND sender_id = ?
        ");
        if (!$check_stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $check_stmt->bind_param("ii", $message_id, $user_id);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        
        if ($check_result->num_rows === 0) {
            sendResponse(false, 'Mesaj bulunamadı veya düzenleme yetkiniz yok');
        }
        
        $message_data = $check_result->fetch_assoc();
        $created_at = strtotime($message_data['created_at']);
        $now = time();
        $hours_diff = ($now - $created_at) / 3600;
        
        if ($hours_diff > 24) {
            sendResponse(false, '24 saatten eski mesajlar düzenlenemez');
        }
        
        // Mesajı güncelle
        $stmt = $conn->prepare("
            UPDATE messages 
            SET message = ?, updated_at = NOW(), is_edited = 1
            WHERE id = ? AND sender_id = ?
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("sii", $new_message, $message_id, $user_id);
        
        if ($stmt->execute()) {
            sendResponse(true, 'Mesaj düzenlendi');
        } else {
            sendResponse(false, 'Mesaj düzenlenemedi: ' . $conn->error);
        }
        
    } else {
        sendResponse(false, 'Geçersiz işlem');
    }

} catch (Exception $e) {
    sendResponse(false, 'Hata: ' . $e->getMessage() . ' - Satır: ' . $e->getLine());
}

$conn->close();
?>

