<?php
// Global error handler
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        header('Content-Type: application/json; charset=utf-8');
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'PHP Fatal Error: ' . $error['message'] . ' in ' . $error['file'] . ' on line ' . $error['line']
        ], JSON_UNESCAPED_UNICODE);
        exit();
    }
});

try {
    require_once '../config.php';
} catch (Exception $e) {
    header('Content-Type: application/json; charset=utf-8');
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Config yükleme hatası: ' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Sadece POST isteği kabul edilir');
}

try {
    $input = getJsonInput();
    if ($input === null) {
        sendResponse(false, 'JSON verisi alınamadı');
    }
    
    $action = sanitize($input['action'] ?? ''); // 'upload', 'get'
    $user_id = intval($input['user_id'] ?? 0);

    if ($user_id <= 0) {
        sendResponse(false, 'Geçersiz kullanıcı ID');
    }

    // Kullanıcı kontrolü
    $user_check = $conn->prepare("SELECT id FROM users WHERE id = ?");
    if (!$user_check) {
        sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
    }
    $user_check->bind_param("i", $user_id);
    $user_check->execute();
    $user_result = $user_check->get_result();

    if ($user_result->num_rows === 0) {
        sendResponse(false, 'Kullanıcı bulunamadı');
    }

    if ($action === 'upload') {
        // Profil görseli yükle (base64)
        $image_data = $input['image_data'] ?? '';
        if (empty($image_data)) {
            sendResponse(false, 'Görsel verisi gereklidir');
        }

        // Base64 verisini kontrol et
        if (!preg_match('/^data:image\/(jpeg|jpg|png|gif);base64,/', $image_data)) {
            sendResponse(false, 'Geçersiz görsel formatı');
        }

        // Profil görselini güncelle
        $stmt = $conn->prepare("
            INSERT INTO user_profiles (user_id, profile_image_data)
            VALUES (?, ?)
            ON DUPLICATE KEY UPDATE
                profile_image_data = VALUES(profile_image_data),
                updated_at = NOW()
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("is", $user_id, $image_data);
        
        if ($stmt->execute()) {
            sendResponse(true, 'Profil görseli yüklendi');
        } else {
            sendResponse(false, 'Profil görseli yüklenemedi: ' . $conn->error);
        }
        
    } elseif ($action === 'get') {
        // Profil görseli getir
        $target_user_id = intval($input['target_user_id'] ?? $user_id);
        
        $stmt = $conn->prepare("
            SELECT profile_image_data 
            FROM user_profiles 
            WHERE user_id = ?
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("i", $target_user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            sendResponse(true, 'Profil görseli bulunamadı', ['image_data' => null]);
        }
        
        $row = $result->fetch_assoc();
        sendResponse(true, 'Profil görseli getirildi', [
            'image_data' => $row['profile_image_data']
        ]);
        
    } elseif ($action === 'delete') {
        // Profil görseli sil
        $stmt = $conn->prepare("
            UPDATE user_profiles 
            SET profile_image_data = NULL,
                updated_at = NOW()
            WHERE user_id = ?
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("i", $user_id);
        
        if ($stmt->execute()) {
            sendResponse(true, 'Profil görseli silindi');
        } else {
            sendResponse(false, 'Profil görseli silinemedi: ' . $conn->error);
        }
        
    } else {
        sendResponse(false, 'Geçersiz işlem');
    }

} catch (Exception $e) {
    sendResponse(false, 'Hata: ' . $e->getMessage() . ' - Satır: ' . $e->getLine());
}

$conn->close();
?>

