<?php
// Hata yakalama için try-catch
try {
    require_once '../config.php';
} catch (Exception $e) {
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'success' => false,
        'message' => 'Config yükleme hatası: ' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Sadece POST isteği kabul edilir');
}

$input = getJsonInput();
$user_id = intval($input['user_id'] ?? 0);
$action = sanitize($input['action'] ?? ''); // 'upload' veya 'download'

if ($user_id <= 0) {
    sendResponse(false, 'Geçersiz kullanıcı ID');
}

// Kullanıcı kontrolü
$user_check = $conn->prepare("SELECT id FROM users WHERE id = ?");
$user_check->bind_param("i", $user_id);
$user_check->execute();
$user_result = $user_check->get_result();

if ($user_result->num_rows === 0) {
    sendResponse(false, 'Kullanıcı bulunamadı');
}

if ($action === 'upload') {
    // Local'den MySQL'e veri yükleme
    $profile = $input['profile'] ?? null;
    $settings = $input['settings'] ?? null;
    $cycle_history = $input['cycle_history'] ?? [];
    $weight_history = $input['weight_history'] ?? [];
    
    // Profil güncelle
    if ($profile) {
        $profile_stmt = $conn->prepare("
            INSERT INTO user_profiles (user_id, name, birth_date, height, profile_image_path)
            VALUES (?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
                name = VALUES(name),
                birth_date = VALUES(birth_date),
                height = VALUES(height),
                profile_image_path = VALUES(profile_image_path),
                updated_at = NOW()
        ");
        $name = sanitize($profile['name'] ?? '');
        $birth_date = !empty($profile['birth_date']) ? sanitize($profile['birth_date']) : null;
        $height = !empty($profile['height']) ? floatval($profile['height']) : null;
        $image_path = sanitize($profile['profile_image_path'] ?? '');
        $profile_stmt->bind_param("issds", $user_id, $name, $birth_date, $height, $image_path);
        $profile_stmt->execute();
    }
    
    // Ayarları güncelle
    if ($settings) {
        $settings_stmt = $conn->prepare("
            INSERT INTO user_settings (
                user_id, last_period_date, cycle_length, period_length,
                theme_mode, language, reminders_enabled, period_reminder,
                ovulation_reminder, reminder_time_hour, reminder_time_minute
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
                last_period_date = VALUES(last_period_date),
                cycle_length = VALUES(cycle_length),
                period_length = VALUES(period_length),
                theme_mode = VALUES(theme_mode),
                language = VALUES(language),
                reminders_enabled = VALUES(reminders_enabled),
                period_reminder = VALUES(period_reminder),
                ovulation_reminder = VALUES(ovulation_reminder),
                reminder_time_hour = VALUES(reminder_time_hour),
                reminder_time_minute = VALUES(reminder_time_minute),
                updated_at = NOW()
        ");
        $last_period = !empty($settings['last_period_date']) ? sanitize($settings['last_period_date']) : null;
        $cycle_len = intval($settings['cycle_length'] ?? 28);
        $period_len = intval($settings['period_length'] ?? 5);
        $theme = sanitize($settings['theme_mode'] ?? 'system');
        $lang = sanitize($settings['language'] ?? 'tr');
        $reminders_enabled = intval($settings['reminders_enabled'] ?? 0);
        $period_reminder = intval($settings['period_reminder'] ?? 0);
        $ovulation_reminder = intval($settings['ovulation_reminder'] ?? 0);
        $reminder_hour = intval($settings['reminder_time_hour'] ?? 9);
        $reminder_minute = intval($settings['reminder_time_minute'] ?? 0);
        
        $settings_stmt->bind_param("isiiissiiiii", $user_id, $last_period, $cycle_len, $period_len,
            $theme, $lang, $reminders_enabled, $period_reminder, $ovulation_reminder,
            $reminder_hour, $reminder_minute);
        $settings_stmt->execute();
    }
    
    // Döngü geçmişini senkronize et
    if (!empty($cycle_history)) {
        foreach ($cycle_history as $cycle) {
            $cycle_stmt = $conn->prepare("
                INSERT INTO cycle_history (user_id, period_start_date, cycle_length, period_length)
                VALUES (?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    cycle_length = VALUES(cycle_length),
                    period_length = VALUES(period_length)
            ");
            $period_date = sanitize($cycle['date']);
            $cycle_length = intval($cycle['cycle_length'] ?? 28);
            $period_length = intval($cycle['period_length'] ?? 5);
            $cycle_stmt->bind_param("isii", $user_id, $period_date, $cycle_length, $period_length);
            $cycle_stmt->execute();
        }
    }
    
    // Kilo geçmişini senkronize et
    if (!empty($weight_history)) {
        foreach ($weight_history as $weight) {
            $weight_stmt = $conn->prepare("
                INSERT INTO weight_history (user_id, weight, record_date, image_path)
                VALUES (?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    weight = VALUES(weight),
                    image_path = VALUES(image_path)
            ");
            $weight_val = floatval($weight['weight']);
            $record_date = sanitize($weight['date']);
            $image_path = sanitize($weight['image_path'] ?? '');
            $weight_stmt->bind_param("idss", $user_id, $weight_val, $record_date, $image_path);
            $weight_stmt->execute();
        }
    }
    
    sendResponse(true, 'Veriler başarıyla senkronize edildi');
    
} elseif ($action === 'download') {
    // MySQL'den local'e veri indirme
    $data = [];
    
    // Profil
    $profile_stmt = $conn->prepare("SELECT * FROM user_profiles WHERE user_id = ?");
    $profile_stmt->bind_param("i", $user_id);
    $profile_stmt->execute();
    $profile_result = $profile_stmt->get_result();
    $data['profile'] = $profile_result->fetch_assoc();
    
    // Ayarlar
    $settings_stmt = $conn->prepare("SELECT * FROM user_settings WHERE user_id = ?");
    $settings_stmt->bind_param("i", $user_id);
    $settings_stmt->execute();
    $settings_result = $settings_stmt->get_result();
    $data['settings'] = $settings_result->fetch_assoc();
    
    // Döngü geçmişi
    $cycle_stmt = $conn->prepare("SELECT * FROM cycle_history WHERE user_id = ? ORDER BY period_start_date DESC");
    $cycle_stmt->bind_param("i", $user_id);
    $cycle_stmt->execute();
    $cycle_result = $cycle_stmt->get_result();
    $data['cycle_history'] = [];
    while ($row = $cycle_result->fetch_assoc()) {
        $data['cycle_history'][] = [
            'date' => $row['period_start_date'],
            'cycle_length' => $row['cycle_length'],
            'period_length' => $row['period_length']
        ];
    }
    
    // Kilo geçmişi
    $weight_stmt = $conn->prepare("SELECT * FROM weight_history WHERE user_id = ? ORDER BY record_date DESC");
    $weight_stmt->bind_param("i", $user_id);
    $weight_stmt->execute();
    $weight_result = $weight_stmt->get_result();
    $data['weight_history'] = [];
    while ($row = $weight_result->fetch_assoc()) {
        $data['weight_history'][] = [
            'weight' => floatval($row['weight']),
            'date' => $row['record_date'],
            'image_path' => $row['image_path']
        ];
    }
    
    sendResponse(true, 'Veriler başarıyla indirildi', $data);
} else {
    sendResponse(false, 'Geçersiz işlem. "upload" veya "download" kullanın');
}

$conn->close();
?>

