<?php
// Global error handler - Fatal error'ları yakalamak için
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        header('Content-Type: application/json; charset=utf-8');
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'PHP Fatal Error: ' . $error['message'] . ' in ' . $error['file'] . ' on line ' . $error['line']
        ], JSON_UNESCAPED_UNICODE);
        exit();
    }
});

// Hata yakalama için try-catch
try {
    require_once '../config.php';
} catch (Exception $e) {
    header('Content-Type: application/json; charset=utf-8');
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Config yükleme hatası: ' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Sadece POST isteği kabul edilir');
}

try {
    $input = getJsonInput();
    if ($input === null) {
        sendResponse(false, 'JSON verisi alınamadı');
    }
    
    $action = sanitize($input['action'] ?? ''); // 'get_profile'
    $user_id = intval($input['user_id'] ?? 0);
    $target_user_id = intval($input['target_user_id'] ?? 0);

    if ($user_id <= 0) {
        sendResponse(false, 'Geçersiz kullanıcı ID');
    }

    if ($target_user_id <= 0) {
        sendResponse(false, 'Geçersiz hedef kullanıcı ID');
    }

    // Kullanıcı kontrolü
    $user_check = $conn->prepare("SELECT id, username, email FROM users WHERE id = ?");
    if (!$user_check) {
        sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
    }
    $user_check->bind_param("i", $user_id);
    $user_check->execute();
    $user_result = $user_check->get_result();

    if ($user_result->num_rows === 0) {
        sendResponse(false, 'Kullanıcı bulunamadı');
    }

    if ($action === 'get_profile') {
        // Kullanıcı profil bilgilerini getir
        $stmt = $conn->prepare("
            SELECT u.id, u.username, u.created_at,
                   up.name as display_name, 
                   up.profile_image_data as profile_image,
                   up.birth_date,
                   up.height
            FROM users u
            LEFT JOIN user_profiles up ON u.id = up.user_id
            WHERE u.id = ?
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("i", $target_user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            sendResponse(false, 'Kullanıcı bulunamadı');
        }
        
        $row = $result->fetch_assoc();
        
        $profile = [
            'id' => $row['id'],
            'username' => $row['username'],
            'display_name' => $row['display_name'],
            'profile_image' => $row['profile_image'],
            'created_at' => $row['created_at'],
            'about' => null, // Hakkında bilgisi şimdilik null, ileride eklenebilir
        ];
        
        sendResponse(true, 'Profil bilgileri getirildi', ['profile' => $profile]);
        
    } elseif ($action === 'update_profile') {
        // Profil bilgilerini güncelle
        $name = sanitize($input['name'] ?? '');
        $birth_date = !empty($input['birth_date']) ? sanitize($input['birth_date']) : null;
        $height = !empty($input['height']) ? floatval($input['height']) : null;
        
        $stmt = $conn->prepare("
            INSERT INTO user_profiles (user_id, name, birth_date, height)
            VALUES (?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
                name = VALUES(name),
                birth_date = VALUES(birth_date),
                height = VALUES(height),
                updated_at = NOW()
        ");
        if (!$stmt) {
            sendResponse(false, 'Veritabanı hatası: ' . $conn->error);
        }
        $stmt->bind_param("issd", $user_id, $name, $birth_date, $height);
        
        if ($stmt->execute()) {
            sendResponse(true, 'Profil güncellendi');
        } else {
            sendResponse(false, 'Profil güncellenemedi: ' . $conn->error);
        }
        
    } else {
        sendResponse(false, 'Geçersiz işlem');
    }

} catch (Exception $e) {
    sendResponse(false, 'Hata: ' . $e->getMessage() . ' - Satır: ' . $e->getLine());
}

$conn->close();
?>

